# -*- coding: utf-8 -*-
"""
Created on Tue Apr 13 08:54:25 2021

@author: s168562

Script to analyze and plot the Holst straining data (electromechanical analysis).

For this version, I tried to improve the looks of the plots:
    - Made them of size 6.5 x 4.5 cm instead of 10 x 7 inch (suitable for use at 100%)
    - Set font size to 8
    - Removed box around legend
    - Removed errors from the gauge factor calculations (GF1_Rini, GF1_R0)
    - Output pdf & tiff
    - Smaller lines
    - Transparent
    -
    - --- still to do:

"""

### Prerequisites
import pandas as pd
import matplotlib.pyplot as plt
import numpy as np
from collections import OrderedDict
import os # for file paths
from math import ceil
import csv
from cycler import cycler # for plot color maps
# import sys
# sys.path.append(os.getcwd())
os.chdir(r"C:\Users\s168562\OneDrive - TU Eindhoven\PhD\1_StretchableGink\Data_analysis\Holst_straining\Analysis")
from cyclic_functions import read_data, calc_cycles
#import ggplot

### Enter file details
headerline = 126
filepath = r"C:\Users\s168562\OneDrive - TU Eindhoven\PhD\1_StretchableGink\Data_analysis\Holst_straining\Raw\August2021_50Percent" # r to escape the backslashes
filename = "20201112-11-11_1000x_2021-08-04_13.45.45#M01.txt"
nr_cycles = 1000
max_strain = 0.5 # maximum strain ratio used (percentage = max_strain*100)
min_strain = 0 # min strain used.
print_thickness = 5.78e-6 # in nm; 5.78e-6 for EU94 & 11.52e-6 for ST604
sample_height = 0.072 # for stress calculations (ST604 = 92 um, EU94 = 72 um)
outpath = r"C:\Users\s168562\OneDrive - TU Eindhoven\PhD\1_StretchableGink\Data_analysis\Holst_straining\Analysis\Figures\Improved"
remove_first_point = False # Set to True to remove the first data point from the averaged data. (== R0)

### Read data
straining_df, Rini, sampleID, mat, length, width, linewidth, outpathfull = read_data(
    filename, filepath, headerline, outpath, nr_cycles, max_strain, 
              min_strain);

### Calculations
(straining_df, straining100_df, straining10_df, strain_series_max, 
            strain_series_max2, strain_series_min, strain_series_min2,
             strain_series_residual_strain
            ) = calc_cycles(
    straining_df, nr_cycles, length, max_strain, min_strain, Rini,
                remove_first_point);

# Save the minimum R value before cycle 1 to use for normalization
Rmin1 = strain_series_min.values[0] 
straining_df['Resist_A_norm'] = straining_df['Resist_A']/Rmin1     

# stress calculation
A = sample_height * width
straining_df['Stress'] = straining_df['Force']/(A)      

### Plots
## Plot prerequisites
straining_df['Labels'] = straining_df['Cycle_nr'].values
straining_df['Labels'] = straining_df['Labels'].astype(str)
plt.style.use('default') # default, ggplot, seaborn-*, dark_background, fivethirtyeight
plt.rcParams['font.size'] = 8 # set plot font size matplotlib
cm = 1/2.54  # convert centimeters to inches
from matplotlib.patches import Patch # for manual legend entries
from matplotlib.lines import Line2D # for manual legend entries
### Default sans-serif fonts:
# plt.rcParams['font.sans-serif'] = ['DejaVu Sans',
#  'Bitstream Vera Sans',
#  'Computer Modern Sans Serif',
#  'Lucida Grande',
#  'Verdana',
#  'Geneva',
#  'Lucid',
#  'Arial',
#  'Helvetica',
#  'Avant Garde',
#  'sans-serif']
plt.rcParams['font.sans-serif'] = ['sans-serif',
                                   'sans-serif',
                                   'DejaVu Sans',
 'Bitstream Vera Sans',
 'Computer Modern Sans Serif',
 'Lucida Grande',
 'Verdana',
 'Geneva',
 'Lucid',
 'Arial',
 'Helvetica',
 'Avant Garde']

#plt.style.available()
# The palette with black:
cbbPalette = ["#000000", "#E69F00", "#56B4E9", "#009E73", "#F0E442", "#0072B2", "#D55E00", "#CC79A7"]
cbPalblue = ["#f0f9e8", "#bae4bc", "#7bccc4", "#43a2ca","#0868ac"]
cbPalblueyellow = ["#ffffcc", "#a1dab4", "#41b6c4", "#2c7fb8","#253494"]
cbPalVivid_correctorder = ["#999999", "#AF1B3F", "#F4A698", "#262626", "#6BAA75", "#255C99", "#7EA3CC"]
cbPalVivid = ["#999999", "#AF1B3F", "#F4A698", "#255C99", "#7EA3CC", "#262626", "#6BAA75"]
TUred = "#C81919"

## set color map (only works when first manually running the script palettes.py stored here:)
# C:\Users\s168562\OneDrive - TU Eindhoven\PhD\Code general\MetBrewer-main\MetBrewer-main\Python\met_brewer
colormap = met_brew(name="OKeeffe1", n=4, brew_type="continuous")
colormap6 = met_brew(name="OKeeffe1", n=6, brew_type="continuous")

# Set font to Latex
# from matplotlib import rc
# rc('font', **{'family':'serif','serif':['Palatino']})
# rc('text', usetex=True)

#### Plot resistance vs cycles
# width = 7*16/9, length = 7 in inches
plt.figure(figsize= [5.5*cm, 3.8*cm])
plt.plot(straining_df['Cycle_nr'], straining_df['Resist_A']/1000, label = r'$R$',
         color = colormap[3], linewidth = 1.5)
ax = plt.axes()
ax.set_xticks([0, 250, 500, 750, 1000])
ax.tick_params(axis="both",direction="in", left="off",labelleft="on", right = "on",
               top = "on")
plt.hlines(y = Rini/1000, xmin = min(straining_df['Cycle_nr']), label = r'$R_0$',
            xmax = max(straining_df['Cycle_nr']), linestyle = 'dashed',
            colors = colormap[2], linewidth = 1.5)
ax.set_facecolor("white")
plt.xlabel(r'Cycle')
plt.ylabel(r'Resistance [k$\Omega$]')
figurepath_pdf = os.path.join(outpathfull, "RvsCycles.pdf")
figurepath_tiff = os.path.join(outpathfull, "RvsCycles.tiff")
plt.savefig(figurepath_tiff, dpi = 300, bbox_inches='tight', transparent = True)
plt.savefig(figurepath_pdf, dpi = 300, bbox_inches='tight', transparent = True)
plt.show()


#### Plot normalized resistance over time
plt.figure(figsize= [5.5*cm, 3.8*cm])
plt.plot(straining_df['Time'], straining_df['Resist_A']/Rmin1,
         color = colormap[3], linewidth = 1.5)
ax = plt.axes()
ax.tick_params(axis="both",direction="in", left="off",labelleft="on", right = "on",
               top = "on")
ax.set_facecolor("white")
ax.set_ylim(1, 30)
ax.set_yticks([1, 10, 20, 30])
plt.xlabel(r'Time [s]')
plt.ylabel(r'$R/R_0$')
figurepath_pdf = os.path.join(outpathfull, "RvsTime_norm.pdf")
figurepath_tiff = os.path.join(outpathfull, "RvsTime_norm.tiff")
plt.savefig(figurepath_tiff, dpi = 300, bbox_inches='tight', transparent = True)
plt.savefig(figurepath_pdf, dpi = 300, bbox_inches='tight', transparent = True)
plt.show()

#### Plot normalized resistance vs cycle
plt.figure(figsize= [5.5*cm, 3.8*cm])
plt.plot(straining_df['Cycle_nr'], straining_df['Resist_A']/Rmin1,
         color = colormap[3], linewidth = 1.5)
ax = plt.axes()
ax.set_ylim(1, 30)
ax.set_yticks([1, 10, 20, 30])
ax.set_xticks([0, 250, 500, 750, 1000])
ax.tick_params(axis="both",direction="in", left="off",labelleft="on", right = "on",
               top = "on")
ax.set_facecolor("white")
plt.xlabel(r'Cycle')
plt.ylabel(r'$R/R_0$')
figurepath_pdf = os.path.join(outpathfull, "RvsCycles_norm.pdf")
figurepath_tiff = os.path.join(outpathfull, "RvsCycles_norm.tiff")
plt.savefig(figurepath_tiff, dpi = 300, bbox_inches='tight', transparent = True)
plt.savefig(figurepath_pdf, dpi = 300, bbox_inches='tight', transparent = True)
plt.show()


### Plot resistance over time
plt.figure(figsize= [5.5*cm, 3.8*cm])
plt.plot(straining_df['Time'], straining_df['Resist_A']/1000, label = r'$R$',
         color = colormap[3], linewidth = 1.5)
ax = plt.axes()
ax.tick_params(axis="both",direction="in", left="off",labelleft="on", right = "on",
               top = "on")
plt.hlines(y = Rini/1000, xmin = min(straining_df['Time']), label = r'$R_0$',
            xmax = max(straining_df['Time']), linestyle = 'dashed',
            colors = colormap[2], linewidth = 1.5)
ax.set_facecolor("white")
plt.xlabel(r'Time [s]')
plt.ylabel(r'Resistance [k$\Omega$]')
figurepath_pdf = os.path.join(outpathfull, "RvsTime.pdf")
figurepath_tiff = os.path.join(outpathfull, "RvsTime.tiff")
plt.savefig(figurepath_tiff, dpi = 300, bbox_inches='tight', transparent = True)
plt.savefig(figurepath_pdf, dpi = 300, bbox_inches='tight', transparent = True)
plt.show()

#### Plot resistance vs cycles
plt.figure(figsize= [5.5*cm, 3.8*cm])
plt.plot(straining_df['Cycle_nr'], straining_df['Resist_A']/1000, label = r'$R$',
         color = colormap[3], linewidth = 1.5)
ax = plt.axes()
ax.set_xticks([0, 250, 500, 750, 1000])
ax.tick_params(axis="both",direction="in", left="off",labelleft="on", right = "on",
               top = "on")
plt.hlines(y = Rini/1000, xmin = min(straining_df['Cycle_nr']), label = r'$R_0$',
            xmax = max(straining_df['Cycle_nr']), linestyle = 'dashed',
            colors = colormap[2], linewidth = 1.5)
ax.set_facecolor("white")
plt.xlabel(r'Cycle')
plt.ylabel(r'Resistance [k$\Omega$]')
figurepath_pdf = os.path.join(outpathfull, "RvsCycles.pdf")
figurepath_tiff = os.path.join(outpathfull, "RvsCycles.tiff")
plt.savefig(figurepath_tiff, dpi = 300, bbox_inches='tight', transparent = True)
plt.savefig(figurepath_pdf, dpi = 300, bbox_inches='tight', transparent = True)
plt.show()


#### Plot normalized resistance over time
plt.figure(figsize= [5.5*cm, 3.8*cm])
plt.plot(straining_df['Time'], straining_df['Resist_A']/Rmin1,
         color = colormap[3], linewidth = 1.5)
ax = plt.axes()
ax.tick_params(axis="both",direction="in", left="off",labelleft="on", right = "on",
               top = "on")
ax.set_facecolor("white")
ax.set_ylim(1, 30)
ax.set_yticks([1, 10, 20, 30])
plt.xlabel(r'Time [s]')
plt.ylabel(r'$R/R_0$')
figurepath_pdf = os.path.join(outpathfull, "RvsTime_norm.pdf")
figurepath_tiff = os.path.join(outpathfull, "RvsTime_norm.tiff")
plt.savefig(figurepath_tiff, dpi = 300, bbox_inches='tight', transparent = True)
plt.savefig(figurepath_pdf, dpi = 300, bbox_inches='tight', transparent = True)
plt.show()

#### Plot normalized resistance vs cycle
plt.figure(figsize= [5.5*cm, 3.8*cm])
plt.plot(straining_df['Cycle_nr'], straining_df['Resist_A']/Rmin1,
         color = colormap[3], linewidth = 1.5)
ax = plt.axes()
ax.set_ylim(1, 30)
ax.set_yticks([1, 10, 20, 30])
ax.set_xticks([0, 250, 500, 750, 1000])
ax.tick_params(axis="both",direction="in", left="off",labelleft="on", right = "on",
               top = "on")
ax.set_facecolor("white")
plt.xlabel(r'Cycle')
plt.ylabel(r'$R/R_0$')
figurepath_pdf = os.path.join(outpathfull, "RvsCycles_norm.pdf")
figurepath_tiff = os.path.join(outpathfull, "RvsCycles_norm.tiff")
plt.savefig(figurepath_tiff, dpi = 300, bbox_inches='tight', transparent = True)
plt.savefig(figurepath_pdf, dpi = 300, bbox_inches='tight', transparent = True)
plt.show()

#### Plot normalized min/max resistance over time
# width = 7*16/9, length = 7 in inches
plt.figure(figsize= [5.5*cm, 3.8*cm])
plt.plot(straining_df['Time'][straining_df['Max_pos']], 
         straining_df['Resist_A'][straining_df['Max_pos']]/Rmin1,
         color = colormap[3], linewidth = 1.5, label = r'50$\%$ strain')
plt.plot(straining_df['Time'][straining_df['Min_pos']], 
         straining_df['Resist_A'][straining_df['Min_pos']]/Rmin1,
         color = colormap[2], linewidth = 1.5, label = r'0$\%$ strain',
         linestyle = 'solid')
ax = plt.axes()
ax.set_ylim(1, 30)
ax.set_yticks([1, 10, 20, 30])
ax.tick_params(axis="both",direction="in", left="off",labelleft="on", right = "on",
               top = "on")
ax.set_facecolor("white")
plt.legend(fontsize = 'small', loc = 'center right', frameon = False)
plt.xlabel(r'Time [s]')
plt.ylabel(r'$R/R_0$')
figurepath_pdf = os.path.join(outpathfull, "RvsTime_minmax_norm.pdf")
figurepath_tiff = os.path.join(outpathfull, "RvsTime_minmax_norm.tiff")
plt.savefig(figurepath_tiff, dpi = 300, bbox_inches='tight', transparent = True)
plt.savefig(figurepath_pdf, dpi = 300, bbox_inches='tight', transparent = True)
plt.show()


#### Plot normalized min/max resistance vs cycle
plt.figure(figsize= [7.2*cm, 4.95*cm])
plt.plot(straining_df['Cycle_nr'][straining_df['Max_pos']], 
         straining_df['Resist_A'][straining_df['Max_pos']]/Rmin1,
         color = colormap[3], linewidth = 2, label = r'50$\%$ strain')
plt.plot(straining_df['Cycle_nr'][straining_df['Min_pos']], 
         straining_df['Resist_A'][straining_df['Min_pos']]/Rmin1,
         color = colormap[2], linewidth = 2, linestyle = 'dashed',
         label = r'0$\%$ strain')
ax = plt.axes()
ax.set_ylim(1, 30)
ax.set_yticks([1, 10, 20, 30])
ax.set_xticks([0, 250, 500, 750, 1000])
ax.tick_params(axis="both",direction="in", left="off",labelleft="on", right = "on",
               top = "on")
ax.set_facecolor("white")
plt.legend(fontsize = 'small', loc = 'center right', frameon = False)
plt.xlabel(r'Cycle')
plt.ylabel(r'$R/R_0$')
figurepath_pdf = os.path.join(outpathfull, "RvsCycles_minmax_norm.pdf")
figurepath_tiff = os.path.join(outpathfull, "RvsCycles_minmax_norm.tiff")
plt.savefig(figurepath_tiff, dpi = 300, bbox_inches='tight', transparent = True)
plt.savefig(figurepath_pdf, dpi = 300, bbox_inches='tight', transparent = True)
plt.show()


### Plot individual cycles
# Cycles 0-5
plotcycles = [0, 1, 2, 3, 4, 5]
plt.figure(figsize= [5.5*cm, 3.8*cm])
plt.plot(straining_df['Time'][straining_df['Cycle_nr'].isin(plotcycles)], 
         straining_df['Resist_A'][straining_df['Cycle_nr'].isin(plotcycles)]/Rmin1,
         color = colormap[3], linewidth = 1.5)
ax = plt.axes()
ax.set_ylim(1, 30)
ax.set_yticks([1, 10, 20, 30])
ax.tick_params(axis="both",direction="in", left="off",labelleft="on", right = "on",
               top = "on")
ax.set_facecolor("white")
plt.xlabel(r'Time [s]')
plt.ylabel(r'$R/R_0$')
figurepath_pdf = os.path.join(outpathfull, "RCycles0-5_norm.pdf")
figurepath_tiff = os.path.join(outpathfull, "RCycles0-5_norm.tiff")
plt.savefig(figurepath_tiff, dpi = 300, bbox_inches='tight', transparent = True)
plt.savefig(figurepath_pdf, dpi = 300, bbox_inches='tight', transparent = True)
plt.show()

# Cycles 0-9
plotcycles = list(range(10))
plt.figure(figsize= [5.5*cm, 3.8*cm])
plt.plot(straining_df['Time'][straining_df['Cycle_nr'].isin(plotcycles)], 
         straining_df['Resist_A'][straining_df['Cycle_nr'].isin(plotcycles)]/Rmin1,
         color = colormap[3], linewidth = 1.5)
ax = plt.axes()
ax.set_ylim(1, 30)
ax.set_yticks([1, 10, 20, 30])
ax.tick_params(axis="both",direction="in", left="off",labelleft="on", right = "on",
               top = "on")
ax.set_facecolor("white")
plt.xlabel(r'Time [s]')
plt.ylabel(r'$R/R_0$')
figurepath_pdf = os.path.join(outpathfull, "RCycles0-9_norm.pdf")
figurepath_tiff = os.path.join(outpathfull, "RCycles0-9_norm.tiff")
plt.savefig(figurepath_tiff, dpi = 300, bbox_inches='tight', transparent = True)
plt.savefig(figurepath_pdf, dpi = 300, bbox_inches='tight', transparent = True)
plt.show()

# plot 5 cycles in continuous palette 2
plt.figure(figsize= [5.5*cm, 3.8*cm])
ax = plt.axes()
ax.set_prop_cycle(cycler('color', colormap6))
plt.plot(straining_df['Cycle_time'][straining_df['Cycle_nr'] == 10], 
         straining_df['Resist_A'][straining_df['Cycle_nr'] == 10]/Rmin1,
         linewidth = 1.5, label = 'Cycle 10')
plt.plot(straining_df['Cycle_time'][straining_df['Cycle_nr'] == 50], 
         straining_df['Resist_A'][straining_df['Cycle_nr'] == 50]/Rmin1,
         linewidth = 1.5, label = 'Cycle 50')
plt.plot(straining_df['Cycle_time'][straining_df['Cycle_nr'] == 100], 
         straining_df['Resist_A'][straining_df['Cycle_nr'] == 100]/Rmin1,
         linewidth = 1.5, label = 'Cycle 100')
plt.plot(straining_df['Cycle_time'][straining_df['Cycle_nr'] == 500], 
         straining_df['Resist_A'][straining_df['Cycle_nr'] == 500]/Rmin1,
         linewidth = 1.5, label = 'Cycle 500')
plt.plot(straining_df['Cycle_time'][straining_df['Cycle_nr'] == 1000], 
         straining_df['Resist_A'][straining_df['Cycle_nr'] == 1000]/Rmin1,
         linewidth = 1.5, label = 'Cycle 1000')
ax.set_xlim(-0.1, 12.1)
ax.set_ylim(1, 30)
ax.set_yticks([1, 10, 20, 30])
ax.tick_params(axis="both",direction="in", left="off",labelleft="on", right = "on",
               top = "on")
ax.set_facecolor("white")
plt.legend(fontsize = 'x-small', loc = 'best', frameon = False, framealpha = 1,
           edgecolor = 'white')
plt.xlabel(r'Time [s]')
plt.ylabel(r'$R/R_0$')
figurepath_pdf = os.path.join(outpathfull, "RCycles5_norm.pdf")
figurepath_tiff = os.path.join(outpathfull, "RCycles5_norm.tiff")
plt.savefig(figurepath_tiff, dpi = 300, bbox_inches='tight', transparent = True)
plt.savefig(figurepath_pdf, dpi = 300, bbox_inches='tight', transparent = True)
plt.show()

### Individual cycles with strain on second axis
## Cycle 999 with strain on second y axis
plotcycles = [999]
plt.figure(figsize= [7.2*cm, 4.95*cm])
fig,ax = plt.subplots(figsize= [7.2*cm, 4.95*cm])
lns1 = ax.plot(straining_df['Cycle_time'][straining_df['Cycle_nr'].isin(plotcycles)], 
         straining_df['Resist_A'][straining_df['Cycle_nr'].isin(plotcycles)]/Rmin1,
         color = colormap[3], linewidth = 2, label = 'Resistance')
ax.tick_params(axis="x",direction="in", left="off", top="on")
ax.tick_params(axis="y",direction="in")
ax.set_facecolor("white")
ax.set_xlabel(r'Time [s]')
ax.set_ylabel(r'$R/R_0$')
ax.set_ylim(1, 30)
ax.set_yticks([1, 10, 20, 30])
# second axis
ax2 = ax.twinx()
lns2 = ax2.plot(straining_df['Cycle_time'][straining_df['Cycle_nr'].isin(plotcycles)], 
         100*straining_df['Strain'][straining_df['Cycle_nr'].isin(plotcycles)],
         color = cbPalVivid[0], linewidth = 2, linestyle = 'dashed', label = 'Strain')
ax2.set_ylabel(r'Strain [%]')
ax2.tick_params(axis="y",direction="in")
ax2.set_yticks([0, 10, 20, 30, 40, 50])
lns = lns1+lns2
labs = [l.get_label() for l in lns]
# ax.legend(lns, labs, loc='lower center', borderaxespad=1, 
#           ncol = 2, bbox_to_anchor = (0.5, -0.4), fontsize = 'small',
#           frameon = False)
plt.legend(lns, labs, loc = 'upper right', fontsize = 'small', 
           frameon = True, framealpha = 1, edgecolor = 'white')
figurepath_pdf = os.path.join(outpathfull, "RCycles999_R_strain.pdf")
figurepath_tiff = os.path.join(outpathfull, "RCycles999_R_strain.tiff")
plt.savefig(figurepath_tiff, dpi = 300, bbox_inches='tight', transparent = True)
plt.savefig(figurepath_pdf, dpi = 300, bbox_inches='tight', transparent = True)
plt.show()

## Cycle 500 with strain on second y axis
plotcycles = [500]
plt.figure(figsize= [7.2*cm, 4.95*cm])
fig,ax = plt.subplots(figsize= [7.2*cm, 4.95*cm])
lns1 = ax.plot(straining_df['Cycle_time'][straining_df['Cycle_nr'].isin(plotcycles)], 
         straining_df['Resist_A'][straining_df['Cycle_nr'].isin(plotcycles)]/Rmin1,
         color = colormap[3], linewidth = 2, label = 'Resistance')
ax.set_ylim(1, 30)
ax.set_yticks([1, 10, 20, 30])
ax.tick_params(axis="x",direction="in", left="off", top="on")
ax.tick_params(axis="y",direction="in")
ax.set_facecolor("white")
ax.set_xlabel(r'Time [s]')
ax.set_ylabel(r'$R/R_0$')
# ax.set_ylim(1.9, 2.6)
# second axis
ax2 = ax.twinx()
lns2 = ax2.plot(straining_df['Cycle_time'][straining_df['Cycle_nr'].isin(plotcycles)], 
         100*straining_df['Strain'][straining_df['Cycle_nr'].isin(plotcycles)],
         color = cbPalVivid[0], linewidth = 2, linestyle = 'dashed', label = 'Strain')
ax2.set_ylabel(r'Strain [%]')
ax2.tick_params(axis="y",direction="in")
ax2.set_yticks([0, 10, 20, 30, 40, 50])
lns = lns1+lns2
labs = [l.get_label() for l in lns]
# ax.legend(lns, labs, loc='lower center', borderaxespad=1, 
#           ncol = 2, bbox_to_anchor = (0.5, -0.3), fontsize = 'small',
#           frameon = False)
plt.legend(lns, labs, loc = 'upper right', fontsize = 'small', 
           frameon = True, framealpha = 1, edgecolor = 'white')
figurepath_pdf = os.path.join(outpathfull, "RCycles500_R_strain.pdf")
figurepath_tiff = os.path.join(outpathfull, "RCycles999_R_strain.tiff")
plt.savefig(figurepath_tiff, dpi = 300, bbox_inches='tight', transparent = True)
plt.savefig(figurepath_pdf, dpi = 300, bbox_inches='tight', transparent = True)
plt.show()

## Cycle 1 with strain on second y axis
plotcycles = [1]
plt.figure(figsize= [7.2*cm, 4.95*cm])
fig,ax = plt.subplots(figsize= [7.2*cm, 4.95*cm])
lns1 = ax.plot(straining_df['Cycle_time'][straining_df['Cycle_nr'].isin(plotcycles)], 
         straining_df['Resist_A'][straining_df['Cycle_nr'].isin(plotcycles)]/Rmin1,
         color = colormap[3], linewidth = 2, label = 'Resistance')
ax.tick_params(axis="x",direction="in", left="off", top="on")
ax.tick_params(axis="y",direction="in")
ax.set_facecolor("white")
ax.set_xlabel(r'Time [s]')
ax.set_ylabel(r'$R/R_0$')
ax.set_ylim(1, 15)
ax.set_yticks([1, 5, 10, 15])
# second axis
ax2 = ax.twinx()
lns2 = ax2.plot(straining_df['Cycle_time'][straining_df['Cycle_nr'].isin(plotcycles)], 
         100*straining_df['Strain'][straining_df['Cycle_nr'].isin(plotcycles)],
         color = cbPalVivid[0], linewidth = 2, linestyle = 'dashed', label = 'Strain')
ax2.set_ylabel(r'Strain [%]')
ax2.tick_params(axis="y",direction="in")
ax2.set_yticks([0, 10, 20, 30, 40, 50])
lns = lns1+lns2
labs = [l.get_label() for l in lns]
# ax.legend(lns, labs, loc='lower center', borderaxespad=1, 
#           ncol = 2, bbox_to_anchor = (0.5, -0.3), fontsize = 'small',
#           frameon = False)
plt.legend(lns, labs, loc = 'upper right', fontsize = 'small', 
           frameon = True, framealpha = 1, edgecolor = 'white')
figurepath_pdf = os.path.join(outpathfull, "RCycles1_R_strain.pdf")
figurepath_tiff = os.path.join(outpathfull, "RCycles1_R_strain.tiff")
plt.savefig(figurepath_tiff, dpi = 300, bbox_inches='tight', transparent = True)
plt.savefig(figurepath_pdf, dpi = 300, bbox_inches='tight', transparent = True)
plt.show()

## Cycle 1-5 with strain on second y axis
plotcycles = [1, 2, 3, 4, 5]
plt.figure(figsize= [7.2*cm, 4.95*cm])
fig,ax = plt.subplots(figsize= [7.2*cm, 4.95*cm])
lns1 = ax.plot(straining_df['Time'][straining_df['Cycle_nr'].isin(plotcycles)], 
         straining_df['Resist_A'][straining_df['Cycle_nr'].isin(plotcycles)]/Rmin1,
         color = colormap[3], linewidth = 2, label = 'Resistance')
ax.tick_params(axis="x",direction="in", left="off", top="on")
ax.tick_params(axis="y",direction="in")
ax.set_facecolor("white")
ax.set_xlabel(r'Time [s]')
ax.set_ylabel(r'$R/R_0$')
ax.set_ylim(1, 25)
ax.set_yticks([1, 5, 10, 15, 20, 25])
# second axis
ax2 = ax.twinx()
lns2 = ax2.plot(straining_df['Time'][straining_df['Cycle_nr'].isin(plotcycles)], 
         100*straining_df['Strain'][straining_df['Cycle_nr'].isin(plotcycles)],
         color = cbPalVivid[0], linewidth = 2, linestyle = 'dashed', label = 'Strain')
ax2.set_ylabel(r'Strain [%]')
ax2.tick_params(axis="y",direction="in")
ax2.set_yticks([0, 10, 20, 30, 40, 50])
lns = lns1+lns2
labs = [l.get_label() for l in lns]
# ax.legend(lns, labs, loc='lower center', borderaxespad=1, 
#           ncol = 2, bbox_to_anchor = (0.5, -0.3), fontsize = 'small',
#           frameon = False)
plt.legend(lns, labs, loc = 'lower right', fontsize = 'small', 
           frameon = True, framealpha = 1, edgecolor = 'white')
# plt.tight_layout() # prevent clipping of axis labels
figurepath_pdf = os.path.join(outpathfull, "RCycles1-5_R_strain.pdf")
figurepath_tiff = os.path.join(outpathfull, "RCycles1-5_R_strain.tiff")
plt.savefig(figurepath_tiff, dpi = 300, bbox_inches='tight', transparent = True)
plt.savefig(figurepath_pdf, dpi = 300, bbox_inches='tight', transparent = True)
plt.savefig(figurepath_pdf, dpi = 300, bbox_inches='tight', transparent = True)
plt.show()


### 3D plots
# plt.rcParams['font.size'] = 8 # set plot font size matplotlib
# non-averaged
## Plot cycle 5 + 500 + 999
# only uphill
firstcycle = 5
middlecycle = 500
lastcycle = 999
fig = plt.figure(figsize= [10*cm, 7*cm])
ax = fig.add_subplot(projection = '3d')

ax.scatter(straining_df['Cycle_nr'][straining_df['Max_pos']][firstcycle-1:-1],
          straining_df['Resist_A'][straining_df['Max_pos']][firstcycle-1:-1]/Rmin1,
          zdir = 'y',
          # zs = 0,
          zs = straining_df['Cycle_time'][straining_df['Max_pos']].values[firstcycle-1],
          color = colormap[3],
          linewidth = 1.5,
          label = r'50$\%$ strain')
ax.scatter(straining_df['Cycle_nr'][straining_df['Min_pos']][firstcycle-1:-1], 
          straining_df['Resist_A'][straining_df['Min_pos']][firstcycle-1:-1]/Rmin1,
          zdir = 'y',
          zs = straining_df['Cycle_time'][straining_df['Min_pos']].values[firstcycle-1],
          color = colormap[1],
          linewidth = 1.5,
          label = r'0$\%$ strain')

ax.plot(straining_df['Cycle_time'][straining_df['Cycle_nr'] == firstcycle], 
         straining_df['Resist_A'][straining_df['Cycle_nr'] == firstcycle]/Rmin1,
         zs = firstcycle,
         zdir = 'x',
         color = colormap[2], linewidth = 2, label = 'Cycle ' + str(firstcycle))

ax.plot(straining_df['Cycle_time'][straining_df['Cycle_nr'] == middlecycle], 
         straining_df['Resist_A'][straining_df['Cycle_nr'] == middlecycle]/Rmin1,
         zs = middlecycle,
         zdir = 'x',
         linestyle = 'dashed',
         color = colormap[2], linewidth = 2, label = 'Cycle ' + str(middlecycle))

ax.plot(straining_df['Cycle_time'][straining_df['Cycle_nr'] == lastcycle], 
         straining_df['Resist_A'][straining_df['Cycle_nr'] == lastcycle]/Rmin1,
         zs = lastcycle,
         zdir = 'x',
         linestyle = 'dotted',
         color = colormap[2], linewidth = 2, label = 'Cycle ' + str(lastcycle))

ax.tick_params(axis="both",direction="in", left="off",labelleft="on", right = "on",
               top = "on")
ax.set_facecolor("white")
# plt.grid()
ax.legend(loc='right', borderaxespad=1, 
          ncol = 1, bbox_to_anchor = (1.65, 0.5), fontsize = 'small',
          frameon = False)
ax.set_xlabel(r'Cycle')
ax.set_ylabel(r'Time [s]')
ax.set_zlabel(r'$R/R_0$')
ax.set_xticks([0, 250, 500, 750, 1000])
ax.tick_params(pad = -4)
# ax.set_zlim(1.5, ceil(max(straining_df['Resist_A'][straining_df['Max_pos']]/Rmin1)))
ax.set_zlim(1, 30)
ax.xaxis.labelpad = -4
ax.yaxis.labelpad = -6
ax.zaxis.labelpad = -8
ax.view_init(elev=20., azim=-45)
figurepath_pdf = os.path.join(outpathfull, "3Dplot_cycle5-500-999_up.pdf")
figurepath_tiff = os.path.join(outpathfull, "3Dplot_cycle5-500-999_up.tiff")
plt.savefig(figurepath_tiff, dpi = 300, bbox_inches='tight', transparent = True)
plt.savefig(figurepath_pdf, dpi = 300, bbox_inches='tight', transparent = True)
plt.show()

### 3D plots
# non-averaged
## Plot cycle 5 + 500 + 999
# only uphill
firstcycle = 5
middlecycle = 500
lastcycle = 999
fig = plt.figure(figsize= [10*cm, 7*cm])
ax = fig.add_subplot(projection = '3d')

ax.scatter(straining_df['Cycle_nr'][straining_df['Max_pos']][firstcycle-1:-1],
          straining_df['Resist_A'][straining_df['Max_pos']][firstcycle-1:-1]/Rmin1,
          zdir = 'y',
          # zs = 0,
          zs = straining_df['Cycle_time'][straining_df['Max_pos']].values[firstcycle-1],
          color = colormap[3],
          linewidth = 1.5,
          label = r'50$\%$ strain')
ax.scatter(straining_df['Cycle_nr'][straining_df['Min_pos']][firstcycle-1:-1], 
          straining_df['Resist_A'][straining_df['Min_pos']][firstcycle-1:-1]/Rmin1,
          zdir = 'y',
          zs = straining_df['Cycle_time'][straining_df['Min_pos']].values[firstcycle-1],
          color = colormap[2],
          linewidth = 1.5,
          label = r'0$\%$ strain')

ax.plot(straining_df['Cycle_time'][straining_df['Cycle_nr'] == firstcycle], 
         straining_df['Resist_A'][straining_df['Cycle_nr'] == firstcycle]/Rmin1,
         zs = firstcycle,
         zdir = 'x',
         color = cbPalVivid[0], linewidth = 2, label = 'Cycle ' + str(firstcycle))

ax.plot(straining_df['Cycle_time'][straining_df['Cycle_nr'] == middlecycle], 
         straining_df['Resist_A'][straining_df['Cycle_nr'] == middlecycle]/Rmin1,
         zs = middlecycle,
         zdir = 'x',
         linestyle = 'dashed',
         color = cbPalVivid[0], linewidth = 2, label = 'Cycle ' + str(middlecycle))

ax.plot(straining_df['Cycle_time'][straining_df['Cycle_nr'] == lastcycle], 
         straining_df['Resist_A'][straining_df['Cycle_nr'] == lastcycle]/Rmin1,
         zs = lastcycle,
         zdir = 'x',
         linestyle = 'dotted',
         color = cbPalVivid[0], linewidth = 2, label = 'Cycle ' + str(lastcycle))

ax.tick_params(axis="both",direction="in", left="off",labelleft="on", right = "on",
               top = "on")
ax.set_facecolor("white")
# plt.grid()
ax.legend(loc='right', borderaxespad=1, 
          ncol = 1, bbox_to_anchor = (1.65, 0.5), fontsize = 'small',
          frameon = False)
ax.set_xlabel(r'Cycle')
ax.set_ylabel(r'Time [s]')
ax.set_zlabel(r'$R/R_0$')
ax.set_xticks([0, 250, 500, 750, 1000])
ax.tick_params(pad = -4)
# ax.set_zlim(1.5, ceil(max(straining_df['Resist_A'][straining_df['Max_pos']]/Rmin1)))
ax.set_zlim(1, 30)
ax.xaxis.labelpad = -4
ax.yaxis.labelpad = -6
ax.zaxis.labelpad = -8
ax.view_init(elev=20., azim=-45)
figurepath_pdf = os.path.join(outpathfull, "3Dplot_cycle5-500-999_up_blue.pdf")
figurepath_tiff = os.path.join(outpathfull, "3Dplot_cycle5-500-999_up_blue.tiff")
plt.savefig(figurepath_tiff, dpi = 300, bbox_inches='tight', transparent = True)
plt.savefig(figurepath_pdf, dpi = 300, bbox_inches='tight', transparent = True)
plt.show()

## Resistance - strain plots
plt.rcParams['font.size'] = 8 # set plot font size matplotlib
plt.figure(figsize= [8*cm, 5.5*cm])
ax = plt.axes()
ax.set_prop_cycle(cycler('color', colormap6))
plt.plot(100*straining_df['Strain'][straining_df['Cycle_nr'] == 1], 
         straining_df['Resist_A'][straining_df['Cycle_nr'] == 1]/1000,
         linewidth = 2, label = 'Cycle 1')
plt.plot(100*straining_df['Strain'][straining_df['Cycle_nr'] == 2], 
         straining_df['Resist_A'][straining_df['Cycle_nr'] == 2]/1000,
         linewidth = 2, label = 'Cycle 2')
plt.plot(100*straining_df['Strain'][straining_df['Cycle_nr'] == 5], 
         straining_df['Resist_A'][straining_df['Cycle_nr'] == 5]/1000,
         linewidth = 2, label = 'Cycle 5')
plt.plot(100*straining_df['Strain'][straining_df['Cycle_nr'] == 100], 
         straining_df['Resist_A'][straining_df['Cycle_nr'] == 100]/1000,
         linewidth = 2, label = 'Cycle 100')
plt.plot(100*straining_df['Strain'][straining_df['Cycle_nr'] == 1000], 
         straining_df['Resist_A'][straining_df['Cycle_nr'] == 1000]/1000,
         linewidth = 2, label = 'Cycle 1000')
# ax.set_ylim(10, 70)
# ax.set_yticks([10, 20, 30, 40, 50, 60, 70])
ax.tick_params(axis="both",direction="in", left="off",labelleft="on", right = "on",
               top = "on")
ax.set_facecolor("white")
plt.legend(fontsize = 'small', loc = 'lower right', frameon = True, framealpha = 1,
           edgecolor = 'white') # solution to have no visible frame but still a white background
plt.xlabel(r'Strain [%]')
plt.ylabel(r'Resistance [k$\Omega$]')
figurepath_pdf = os.path.join(outpathfull, "R-strain.pdf")
figurepath_tiff = os.path.join(outpathfull, "R-strain.tiff")
plt.savefig(figurepath_tiff, dpi = 300, bbox_inches='tight', transparent = True)
plt.savefig(figurepath_pdf, dpi = 300, bbox_inches='tight', transparent = True)
plt.show()

# only first cycle
plt.figure(figsize= [8*cm, 5.5*cm])
plt.plot(100*straining_df['Strain'][straining_df['Cycle_nr'] == 1], 
         straining_df['Resist_A'][straining_df['Cycle_nr'] == 1]/1000,
         color = colormap[3], linewidth = 2)
ax = plt.axes()
ax.tick_params(axis="both",direction="in", left="off",labelleft="on", right = "on",
               top = "on")
ax.set_facecolor("white")
plt.xlabel(r'Strain [%]')
plt.ylabel(r'Resistance [k$\Omega$]')
figurepath_pdf = os.path.join(outpathfull, "R-strain_cycle1.pdf")
figurepath_tiff = os.path.join(outpathfull, "R-strain_cycle1.tiff")
plt.savefig(figurepath_tiff, dpi = 300, bbox_inches='tight', transparent = True)
plt.savefig(figurepath_pdf, dpi = 300, bbox_inches='tight', transparent = True)
plt.show()

# only first cycle - y10-50
plt.figure(figsize= [8*cm, 5.5*cm])
plt.plot(100*straining_df['Strain'][straining_df['Cycle_nr'] == 1], 
         straining_df['Resist_A'][straining_df['Cycle_nr'] == 1]/1000,
         color = colormap[3], linewidth = 2)
ax = plt.axes()
ax.tick_params(axis="both",direction="in", left="off",labelleft="on", right = "on",
               top = "on")
ax.set_facecolor("white")
plt.xlabel(r'Strain [%]')
plt.ylabel(r'Resistance [k$\Omega$]')
figurepath_pdf = os.path.join(outpathfull, "R-strain_cycle1_y10-50.pdf")
figurepath_tiff = os.path.join(outpathfull, "R-strain_cycle1_y10-50.tiff")
plt.savefig(figurepath_tiff, dpi = 300, bbox_inches='tight', transparent = True)
plt.savefig(figurepath_pdf, dpi = 300, bbox_inches='tight', transparent = True)
plt.show()

# only second cycle
plt.figure(figsize= [8*cm, 5.5*cm])
plt.plot(100*straining_df['Strain'][straining_df['Cycle_nr'] == 2], 
         straining_df['Resist_A'][straining_df['Cycle_nr'] == 2]/1000,
         color = colormap[3], linewidth = 2)
ax = plt.axes()
ax.tick_params(axis="both",direction="in", left="off",labelleft="on", right = "on",
               top = "on")
ax.set_facecolor("white")
plt.xlabel(r'Strain [%]')
plt.ylabel(r'Resistance [k$\Omega$]')
figurepath_pdf = os.path.join(outpathfull, "R-strain_cycle2.pdf")
figurepath_tiff = os.path.join(outpathfull, "R-strain_cycle2.tiff")
plt.savefig(figurepath_tiff, dpi = 300, bbox_inches='tight', transparent = True)
plt.savefig(figurepath_pdf, dpi = 300, bbox_inches='tight', transparent = True)
plt.show()

# only cycle 10
plt.figure(figsize= [8*cm, 5.5*cm])
plt.plot(100*straining_df['Strain'][straining_df['Cycle_nr'] == 10], 
         straining_df['Resist_A'][straining_df['Cycle_nr'] == 10]/1000,
         color = colormap[3], linewidth = 2)
ax = plt.axes()
ax.tick_params(axis="both",direction="in", left="off",labelleft="on", right = "on",
               top = "on")
ax.set_facecolor("white")
plt.xlabel(r'Strain [%]')
plt.ylabel(r'Resistance [k$\Omega$]')
figurepath_pdf = os.path.join(outpathfull, "R-strain_cycle10.pdf")
figurepath_tiff = os.path.join(outpathfull, "R-strain_cycle10.tiff")
plt.savefig(figurepath_tiff, dpi = 300, bbox_inches='tight', transparent = True)
plt.savefig(figurepath_pdf, dpi = 300, bbox_inches='tight', transparent = True)
plt.show()

# only cycle 100
plt.figure(figsize= [8*cm, 5.5*cm])
plt.plot(100*straining_df['Strain'][straining_df['Cycle_nr'] == 100], 
         straining_df['Resist_A'][straining_df['Cycle_nr'] == 100]/1000,
         color = colormap[3], linewidth = 2)
ax = plt.axes()
ax.tick_params(axis="both",direction="in", left="off",labelleft="on", right = "on",
               top = "on")
ax.set_facecolor("white")
plt.xlabel(r'Strain [%]')
plt.ylabel(r'Resistance [k$\Omega$]')
figurepath_pdf = os.path.join(outpathfull, "R-strain_cycle100.pdf")
figurepath_tiff = os.path.join(outpathfull, "R-strain_cycle100.tiff")
plt.savefig(figurepath_tiff, dpi = 300, bbox_inches='tight', transparent = True)
plt.savefig(figurepath_pdf, dpi = 300, bbox_inches='tight', transparent = True)
plt.show()

# only cycle 500
plt.figure(figsize= [8*cm, 5.5*cm])
plt.plot(100*straining_df['Strain'][straining_df['Cycle_nr'] == 500], 
         straining_df['Resist_A'][straining_df['Cycle_nr'] == 500]/1000,
         color = colormap[3], linewidth = 2)
ax = plt.axes()
ax.tick_params(axis="both",direction="in", left="off",labelleft="on", right = "on",
               top = "on")
ax.set_facecolor("white")
plt.xlabel(r'Strain [%]')
plt.ylabel(r'Resistance [k$\Omega$]')
figurepath_pdf = os.path.join(outpathfull, "R-strain_cycle500.pdf")
figurepath_tiff = os.path.join(outpathfull, "R-strain_cycle500.tiff")
plt.savefig(figurepath_tiff, dpi = 300, bbox_inches='tight', transparent = True)
plt.savefig(figurepath_pdf, dpi = 300, bbox_inches='tight', transparent = True)
plt.show()

# only cycle 1000
plt.figure(figsize= [8*cm, 5.5*cm])
plt.plot(100*straining_df['Strain'][straining_df['Cycle_nr'] == 1000], 
         straining_df['Resist_A'][straining_df['Cycle_nr'] ==1000]/1000,
         color = colormap[3], linewidth = 2)
ax = plt.axes()
ax.tick_params(axis="both",direction="in", left="off",labelleft="on", right = "on",
               top = "on")
ax.set_facecolor("white")
plt.xlabel(r'Strain [%]')
plt.ylabel(r'Resistance [k$\Omega$]')
figurepath_pdf = os.path.join(outpathfull, "R-strain_cycle10000.pdf")
figurepath_tiff = os.path.join(outpathfull, "R-strain_cycle1000.tiff")
plt.savefig(figurepath_tiff, dpi = 300, bbox_inches='tight', transparent = True)
plt.savefig(figurepath_pdf, dpi = 300, bbox_inches='tight', transparent = True)
plt.show()

#### Residual strain vs cycle ####
plt.figure(figsize= [5.5*cm, 3.8*cm])
plt.plot(list(range(1,1001)), 100*strain_series_residual_strain, label = r'$R$',
         color = colormap[3], linewidth = 1.5)
ax = plt.axes()
ax.set_xticks([0, 250, 500, 750, 1000])
ax.tick_params(axis="both",direction="in", left="off",labelleft="on", right = "on",
               top = "on")
ax.set_facecolor("white")
plt.xlabel(r'Cycle')
plt.ylabel(r'Residual strain [%]')
figurepath_pdf = os.path.join(outpathfull, "ResStrainvsCycles.pdf")
figurepath_tiff = os.path.join(outpathfull, "ResStrainvsCycles.tiff")
plt.savefig(figurepath_tiff, 
            dpi = 300, bbox_inches='tight', transparent = True)
plt.savefig(figurepath_pdf, dpi = 300, bbox_inches='tight', transparent = True)
plt.show()

#### Residual strain vs cycle y0-10 ####
plt.figure(figsize= [5.5*cm, 3.8*cm])
plt.plot(list(range(1,1001)), 100*strain_series_residual_strain, label = r'$R$',
         color = colormap[3], linewidth = 1.5)
ax = plt.axes()
ax.set_ylim(0,10)
# ax.set_xlim(0,1000)
ax.set_xticks([0, 250, 500, 750, 1000])
ax.tick_params(axis="both",direction="in", left="off",labelleft="on", right = "on",
               top = "on")
ax.set_facecolor("white")
plt.xlabel(r'Cycle')
plt.ylabel(r'Residual strain [%]')
figurepath_pdf = os.path.join(outpathfull, "ResStrainvsCycles_y0-10.pdf")
figurepath_tiff = os.path.join(outpathfull, "ResStrainvsCycles_y0-10.tiff")
plt.savefig(figurepath_tiff, 
            dpi = 300, bbox_inches='tight', transparent = True)
plt.savefig(figurepath_pdf, dpi = 300, bbox_inches='tight', transparent = True)
plt.show()

#### effective strain vs cycle ####
effective_strain = list(100*(max_strain - strain_series_residual_strain[0:-1]))
effective_strain.insert(0, 100*max_strain)
plt.figure(figsize= [5.5*cm, 3.8*cm])
plt.plot(list(range(1,1001)), effective_strain, label = r'$R$',
         color = colormap[3], linewidth = 1.5)
ax = plt.axes()
ax.set_xticks([0, 250, 500, 750, 1000])
ax.tick_params(axis="both",direction="in", left="off",labelleft="on", right = "on",
               top = "on")
ax.set_facecolor("white")
plt.xlabel(r'Cycle')
plt.ylabel(r'Effective strain [%]')
figurepath_pdf = os.path.join(outpathfull, "EffectiveStrainvsCycles.pdf")
figurepath_tiff = os.path.join(outpathfull, "EffectiveStrainvsCycles.tiff")
plt.savefig(figurepath_tiff, 
            dpi = 300, bbox_inches='tight', transparent = True)
plt.savefig(figurepath_pdf, dpi = 300, bbox_inches='tight', transparent = True)
plt.show()

#### effective strain vs cycle y 0-50 ####
plt.figure(figsize= [5.5*cm, 3.8*cm])
plt.plot(list(range(1,1001)), effective_strain, label = r'$R$',
         color = colormap[3], linewidth = 1.5)
ax = plt.axes()
ax.set_ylim(0,50)
ax.set_xticks([0, 250, 500, 750, 1000])
ax.tick_params(axis="both",direction="in", left="off",labelleft="on", right = "on",
               top = "on")
ax.set_facecolor("white")
plt.xlabel(r'Cycle')
plt.ylabel(r'Effective strain [%]')
figurepath_pdf = os.path.join(outpathfull, "EffectiveStrainvsCycles_y0-20.pdf")
figurepath_tiff = os.path.join(outpathfull, "EffectiveStrainvsCyclesy0-20.tiff")
plt.savefig(figurepath_tiff, 
            dpi = 300, bbox_inches='tight', transparent = True)
plt.savefig(figurepath_pdf, dpi = 300, bbox_inches='tight', transparent = True)
plt.show()


### Gauge factor calculations (3 methods)
### Gauge factor calculations (dynamic gauge factor as found in the literature (lynch2020graphene), but one could argue that taking a fixed max. strain rather than the effective strain distorts the results)
GF1 = (strain_series_max.values[0]  - strain_series_min.values[0])/(strain_series_min.values[0]*max_strain)
GF2 = (strain_series_max.values[1] - strain_series_min.values[1])/(strain_series_min.values[1]*max_strain) # within-cycle gauge factor cycle 2
GF10 = (strain_series_max.values[9] - strain_series_min.values[9])/(strain_series_min.values[9]*max_strain) # within-cycle gauge factor cycle 10
GF50 = (strain_series_max.values[49] - strain_series_min.values[49])/(strain_series_min.values[49]*max_strain) # within-cycle gauge factor cycle 50
GF100 = (strain_series_max.values[99] - strain_series_min.values[99])/(strain_series_min.values[99]*max_strain) # within-cycle gauge factor cycle 100
GF200 = (strain_series_max.values[199] - strain_series_min.values[199])/(strain_series_min.values[199]*max_strain) # within-cycle gauge factor cycle 100
GF300 = (strain_series_max.values[299] - strain_series_min.values[299])/(strain_series_min.values[299]*max_strain) # within-cycle gauge factor cycle 100
GF400 = (strain_series_max.values[399] - strain_series_min.values[399])/(strain_series_min.values[399]*max_strain) # within-cycle gauge factor cycle 100
GF500 = (strain_series_max.values[499] - strain_series_min.values[499])/(strain_series_min.values[499]*max_strain) # within-cycle gauge factor cycle 100
GF600 = (strain_series_max.values[599] - strain_series_min.values[599])/(strain_series_min.values[599]*max_strain) # within-cycle gauge factor cycle 100
GF700 = (strain_series_max.values[699] - strain_series_min.values[699])/(strain_series_min.values[699]*max_strain) # within-cycle gauge factor cycle 100
GF800 = (strain_series_max.values[799] - strain_series_min.values[799])/(strain_series_min.values[799]*max_strain) # within-cycle gauge factor cycle 100
GF900 = (strain_series_max.values[899] - strain_series_min.values[899])/(strain_series_min.values[899]*max_strain) # within-cycle gauge factor cycle 900
GF999 = (strain_series_max.values[998] - strain_series_min.values[998])/(strain_series_min.values[998]*max_strain) # within-cycle gauge factor cycle 999
GF1000 = (strain_series_max.values[999] - strain_series_min.values[999])/(strain_series_min.values[999]*max_strain) # within-cycle gauge factor cycle 1000

### Gauge factor calculations normalized by R0 instead of Rmin (regular gauge factor)
GF1_Rini = (strain_series_max.values[0] - Rini)/(Rini*max_strain) # initial gauge factor compared to before backing removal
GF2_Rini = (strain_series_max.values[1] - Rini)/(Rini*max_strain) # gauge factor cycle 2
GF10_Rini = (strain_series_max.values[9] - Rini)/(Rini*max_strain) # gauge factor cycle 10
GF50_Rini = (strain_series_max.values[49] - Rini)/(Rini*max_strain) # gauge factor cycle 50
GF100_Rini = (strain_series_max.values[99] - Rini)/(Rini*max_strain) #  gauge factor cycle 100
GF200_Rini = (strain_series_max.values[199] - Rini)/(Rini*max_strain) #  gauge factor cycle 100
GF300_Rini = (strain_series_max.values[299] - Rini)/(Rini*max_strain) #  gauge factor cycle 100
GF400_Rini = (strain_series_max.values[399] - Rini)/(Rini*max_strain) #  gauge factor cycle 100
GF500_Rini = (strain_series_max.values[499] - Rini)/(Rini*max_strain) #  gauge factor cycle 100
GF600_Rini = (strain_series_max.values[599] - Rini)/(Rini*max_strain) #  gauge factor cycle 100
GF700_Rini = (strain_series_max.values[699] - Rini)/(Rini*max_strain) #  gauge factor cycle 100
GF800_Rini = (strain_series_max.values[799] - Rini)/(Rini*max_strain) #  gauge factor cycle 100
GF900_Rini = (strain_series_max.values[899] - Rini)/(Rini*max_strain) # gauge factor cycle 900
GF999_Rini = (strain_series_max.values[998] - Rini)/(Rini*max_strain) #  gauge factor cycle 999
GF1000_Rini = (strain_series_max.values[999] - Rini)/(Rini*max_strain) #  gauge factor cycle 1000

### Gauge factor calculations normalized by Rmin and effective strain (dynamic gauge factor best method)
GF1_eff = (strain_series_max.values[0]  - strain_series_min.values[0])/(strain_series_min.values[0]*effective_strain[0]/100)
GF2_eff = (strain_series_max.values[1] - strain_series_min.values[1])/(strain_series_min.values[1]*effective_strain[1]/100) # within-cycle gauge factor cycle 2
GF10_eff = (strain_series_max.values[9] - strain_series_min.values[9])/(strain_series_min.values[9]*effective_strain[9]/100) # within-cycle gauge factor cycle 10
GF50_eff = (strain_series_max.values[49] - strain_series_min.values[49])/(strain_series_min.values[49]*effective_strain[49]/100) # within-cycle gauge factor cycle 50
GF100_eff = (strain_series_max.values[99] - strain_series_min.values[99])/(strain_series_min.values[99]*effective_strain[99]/100) # within-cycle gauge factor cycle 100
GF200_eff = (strain_series_max.values[199] - strain_series_min.values[199])/(strain_series_min.values[199]*effective_strain[199]/100) # within-cycle gauge factor cycle 100
GF300_eff = (strain_series_max.values[299] - strain_series_min.values[299])/(strain_series_min.values[299]*effective_strain[299]/100) # within-cycle gauge factor cycle 100
GF400_eff = (strain_series_max.values[399] - strain_series_min.values[399])/(strain_series_min.values[399]*effective_strain[399]/100) # within-cycle gauge factor cycle 100
GF500_eff = (strain_series_max.values[499] - strain_series_min.values[499])/(strain_series_min.values[499]*effective_strain[499]/100) # within-cycle gauge factor cycle 100
GF600_eff = (strain_series_max.values[599] - strain_series_min.values[599])/(strain_series_min.values[599]*effective_strain[599]/100) # within-cycle gauge factor cycle 100
GF700_eff = (strain_series_max.values[699] - strain_series_min.values[699])/(strain_series_min.values[699]*effective_strain[699]/100) # within-cycle gauge factor cycle 100
GF800_eff = (strain_series_max.values[799] - strain_series_min.values[799])/(strain_series_min.values[799]*effective_strain[799]/100) # within-cycle gauge factor cycle 100
GF900_eff = (strain_series_max.values[899] - strain_series_min.values[899])/(strain_series_min.values[899]*effective_strain[899]/100) # within-cycle gauge factor cycle 900
GF999_eff = (strain_series_max.values[998] - strain_series_min.values[998])/(strain_series_min.values[998]*effective_strain[998]/100) # within-cycle gauge factor cycle 999
GF1000_eff = (strain_series_max.values[999] - strain_series_min.values[999])/(strain_series_min.values[999]*effective_strain[999]/100) # within-cycle gauge factor cycle 1000

#R/Rmin1 after 300s relaxation
recovery = straining_df['Resist_A_norm'].iloc[-1]

# Calculate sheet resistances in Ohm normalized to 25 um
# There are 76 squares of 1x1 mm2 in a 76x1 mm dogbone.
Rs = (Rini/(76))*print_thickness/25e-6

# Write gauge factor data to csv file
csvpath = os.path.join(outpathfull, "gaugefactors.csv")
header = ['Method','GF1', 'GF2', 'GF10', 'GF50','GF100', 'GF200', 
          'GF300', 'GF400', 'GF500', 'GF600', 'GF700', 'GF800','GF900', 'GF999', 
          'GF1000', 'R0prepostprocessing', 'Rini', 'Rmin_1', 
          'Sheet_res_ini', 'Recovery_300s_relax']
data1 = ['Rmin_maxstrain',GF1, GF2, GF10, GF50, GF100, GF200, GF300, GF400, GF500,
        GF600, GF700, GF800, GF900, GF999, GF1000, Rini, Rini,
        strain_series_min.values[0], Rs, recovery]
data2 = ['Rini_maxstrain', GF1_Rini, GF2_Rini, GF10_Rini, 
         GF50_Rini, GF100_Rini, GF200_Rini, GF300_Rini, GF400_Rini, GF500_Rini,
         GF600_Rini, GF700_Rini, GF800_Rini, GF900_Rini, GF999_Rini, GF1000_Rini, 
         Rini, Rini, strain_series_min.values[0], Rs, recovery]
data3 = ['Rmin_effectivestrain', GF1_eff, GF2_eff, GF10_eff, 
         GF50_eff, GF100_eff, GF200_eff, GF300_eff, GF400_eff, GF500_eff,
         GF600_eff, GF700_eff, GF800_eff, GF900_eff, GF999_eff, GF1000_eff, 
         Rini, Rini, strain_series_min.values[0], Rs, recovery]

with open(csvpath, 'w', newline='') as f:
    writer = csv.writer(f)
    writer.writerow(header)
    writer.writerow(data1)
    writer.writerow(data2)
    writer.writerow(data3)

# Write min/max & effective strain data to csv file
csvpath = os.path.join(outpathfull, "peaks_valleys.csv")
header = ['Cycle', 'Min1', 'Max1', 'Max2', 'Min2', 'Effective_strain']
data = [tuple(range(1,1001)), tuple(strain_series_min.values), tuple(strain_series_max.values),
        tuple(strain_series_max2.values), tuple(strain_series_min2.values), tuple(effective_strain)]
data_t = list(zip(*data))
with open(csvpath, 'w', newline='') as f:
    writer = csv.writer(f)
    writer.writerow(header)
    writer.writerows(data_t)